#!/usr/bin/env python3
"""
Viveka Instagram Analyzer - Simple utility to analyze Instagram profiles with AI.

EXECUTION: This script is executed directly by Claude, not loaded into context.

Usage:
    python analyze.py <username> <prompt> [--max-posts N]

Example:
    python analyze.py natgeo "What are their main interests?" --max-posts 15
"""

import argparse
import sys
import time
from typing import Optional

import requests

API_BASE_URL = "https://viveka.darkgravitylabs.com"
DEFAULT_MAX_POSTS = 15
DEFAULT_POLL_ATTEMPTS = 100
# AIDEV-NOTE: 5-second interval balances responsiveness with API load.
# 100 attempts at 5s = ~500s max wait = reasonable for 1-2 min typical jobs
POLL_INTERVAL_SECONDS = 5


def main():
    """CLI entry point."""
    parser = argparse.ArgumentParser(
        description="Analyze Instagram profiles with AI",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  python analyze.py natgeo "What are their main interests?"
  python analyze.py natgeo --max-posts 10 "What are their main interests?"
        """,
    )

    parser.add_argument(
        "username",
        help="Instagram URL or username (without @)",
    )
    parser.add_argument(
        "prompt",
        help="Question or instruction for AI analysis",
    )
    parser.add_argument(
        "--max-posts",
        type=int,
        default=DEFAULT_MAX_POSTS,
        metavar="N",
        help=f"Number of posts to analyze (default: {DEFAULT_MAX_POSTS}, max: 100)",
    )

    args = parser.parse_args()

    analysis = analyze_profile(args.username, args.prompt, args.max_posts)

    if analysis:
        print(analysis)  # Print analysis to stdout
        sys.exit(0)
    else:
        print("[ERROR] Analysis failed", file=sys.stderr)
        sys.exit(1)


def analyze_profile(
    username: str,
    prompt: str,
    max_posts: int = DEFAULT_MAX_POSTS,
    max_attempts: int = DEFAULT_POLL_ATTEMPTS,
) -> Optional[str]:
    """
    Analyze an Instagram profile with AI.

    Args:
        username: Instagram username (without @)
        prompt: Question or instruction for AI analysis
        max_posts: Number of posts to analyze (default: 15)
        max_attempts: Maximum polling attempts (default: 100)

    Returns:
        Analysis text if successful, None if failed
    """
    # Submit job
    try:
        response = requests.post(
            f"{API_BASE_URL}/api/v1/jobs",
            json={
                "profiles": [username],
                "prompt": prompt,
                "max_posts_per_profile": max_posts,
            },
            timeout=10,
        )
        response.raise_for_status()
        job_id = response.json()["job_id"]
    except requests.exceptions.RequestException as e:
        print(f"[ERROR] Failed to submit job: {e}", file=sys.stderr)
        return None

    # Poll for completion
    for attempt in range(1, max_attempts + 1):
        try:
            response = requests.get(
                f"{API_BASE_URL}/api/v1/jobs/{job_id}",
                timeout=10,
            )
            response.raise_for_status()
            job_data = response.json()
            status = job_data["status"]

            if status == "failed":
                error = job_data.get("error", "Unknown error")
                print(f"[ERROR] Job failed: {error}", file=sys.stderr)
                return None

            if status == "completed":
                result = job_data["results"][0]
                if result["status"] != "success":
                    error_code = result.get("error_code", "UNKNOWN")
                    error_message = result.get("error_message", "No error message")
                    print(
                        f"[ERROR] Analysis failed: [{error_code}] {error_message}",
                        file=sys.stderr,
                    )
                    return None
                return result["analysis"]

            time.sleep(POLL_INTERVAL_SECONDS)

        except requests.exceptions.RequestException as e:
            print(f"[ERROR] Polling failed on attempt {attempt}: {e}", file=sys.stderr)
            return None

    print(f"[ERROR] Timeout after {max_attempts} attempts", file=sys.stderr)
    return None


if __name__ == "__main__":
    main()
