# Viveka API Reference

Complete API documentation for the Viveka Instagram Analyzer service.

## Table of Contents

1. API Endpoint
2. How It Works
3. Step 1: Submit Analysis Job
4. Step 2: Poll for Results
5. Step 3: Extract Results
6. Error Codes
7. Rate Limits
8. Complete Example

## API Endpoint

**Base URL**: `https://viveka.darkgravitylabs.com`

**Submit Job**: `POST /api/v1/jobs`
**Check Status**: `GET /api/v1/jobs/{job_id}`

## How It Works

1. **Submit Analysis Job**: POST request to `/api/v1/jobs` with username and prompt
2. **Poll for Results**: GET request to `/api/v1/jobs/{job_id}` to check status
3. **Retrieve Analysis**: Once status is "completed", results contain the AI analysis

The analysis typically takes 1-2 minutes depending on the number of posts.

## Step 1: Submit Analysis Job

Make a POST request to submit the analysis:

```python
import requests

response = requests.post(
    "https://viveka.darkgravitylabs.com/api/v1/jobs",
    json={
        "profiles": ["username"],  # Instagram username (without @)
        "prompt": "Your question here",
        "max_posts_per_profile": 15  # Optional: 0-100
    }
)

job_data = response.json()
job_id = job_data["job_id"]
```

### Request Fields

- `profiles`: List of Instagram usernames (currently only 1 supported)
- `prompt`: Your question/instruction for the AI
- `max_posts_per_profile`: Number of posts to analyze (default: 20, max: 100)

### Response

```json
{
    "job_id": "550e8400-e29b-41d4-a716-446655440000"
}
```

## Step 2: Poll for Results

The job runs asynchronously. Poll the status endpoint until completion:

```python
import time

while True:
    status_response = requests.get(
        f"https://viveka.darkgravitylabs.com/api/v1/jobs/{job_id}"
    )
    job_status = status_response.json()

    if job_status["status"] in ["completed", "failed"]:
        break

    time.sleep(5)  # Wait 5 seconds between polls
```

### Status Values

- `pending`: Job queued, not started yet
- `running`: Currently analyzing profile
- `completed`: Analysis finished successfully
- `failed`: Error occurred during analysis

## Step 3: Extract Results

Once completed, extract the analysis from the results:

```python
if job_status["status"] == "completed":
    result = job_status["results"][0]

    if result["status"] == "success":
        analysis = result["analysis"]
        print(analysis)
    else:
        # Handle profile-level errors
        print(f"Error: {result['error_message']}")
else:
    # Handle job-level errors
    print(f"Job failed: {job_status.get('error')}")
```

### Result Structure

```json
{
    "job_id": "550e8400-e29b-41d4-a716-446655440000",
    "status": "completed",
    "created_at": "2025-01-01T00:00:00Z",
    "results": [
        {
            "username": "example",
            "status": "success",
            "analysis": "AI analysis text here..."
        }
    ]
}
```

### Error Response Structure

```json
{
    "job_id": "550e8400-e29b-41d4-a716-446655440000",
    "status": "completed",
    "created_at": "2025-01-01T00:00:00Z",
    "results": [
        {
            "username": "example",
            "status": "error",
            "error_code": "PRIVATE_PROFILE",
            "error_message": "This profile is private and cannot be analyzed"
        }
    ]
}
```

## Error Codes

### Profile-Level Errors

These appear in the `results` array when a specific profile fails:

- `PROFILE_NOT_FOUND`: Instagram username doesn't exist
- `PRIVATE_PROFILE`: Profile is private and cannot be analyzed
- `CONTENT_BLOCKED`: Analysis blocked by AI safety filters
- `EMPTY_ANALYSIS`: AI returned no analysis (try different prompt)
- `APIFY_ERROR`: Failed to fetch Instagram data
- `GEMINI_ERROR`: AI analysis service error
- `UNKNOWN_ERROR`: Unexpected error occurred

### Job-Level Errors

These appear when the entire job fails:

- `NOT_IMPLEMENTED`: Requested feature not available yet (e.g., multiple profiles)
- HTTP 404: Job not found or expired
- HTTP 429: Rate limit exceeded

## Rate Limits

- 2 requests per minute per IP address
- Jobs expire after 24 hours

## Complete Example
See the `./analyze.py` file for a complete example with error handling and polling logic.
